using System;
using System.Timers;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.HL7AL
{
	/// <summary>
	/// Delegate used by blocking timer to call a single method when timer is elapsed.
	/// </summary>
	public delegate void BlockingTimerCallback();

	/// <summary>
	/// This class wraps standard server timer implementing blocking behavior. The differences 
	/// from are standard server timer are:
	/// 1) Start/stop/elapsed methods are interlocked: if a call to one of these methods
	/// occurs while an other is executed, it will wait for first method to complete.
	/// 2) When timer elapses only one supplied method is called via delegate.
	/// 3) Supplied callback method will never be called after timer is stopped. 
	/// 4) Only one thread will call a callback method when timer elapses. 
	/// 5) When timer elapses, it's stopped before calling callback method. If AutoReset
	/// is set to true, then timer will be restarted after callback method is finished. 
	/// Hence, if timer interval is 1 second, autoreset is set to true and callback method 
	/// takes 2 seconds to run, actual interval between calls to callback method will be 
	/// 3 seconds (2+1: timer elapsed - start count, call back is run - 2 seconds, timer 
	/// elapsed - 1 second - end count).
	/// 6) This class is thread safe.
	/// 
	/// This timer is only recommended for use with large intervals (more than 1 second) and 
	/// callback methods that take at least ten times less time to run than 1 interval duration. 
	/// </summary>
	public class BlockingTimer 
	{
		private readonly BlockingTimerCallback _elapsedCallback;
		private readonly System.Timers.Timer _timer;
		bool _stoppedFlag;

		/// <summary>
		/// Simple constructor setting all parameters needed to use timer. 
		/// </summary>
		/// <param name="interval">Timer interval.</param>
		/// <param name="autoReset">Flag indicating if timer should provide recurrent 'ticks'.</param>
		/// <param name="elapsedCallback">Callback method that will be called when interval elapses.</param>
		public BlockingTimer( Timeout interval, bool autoReset, BlockingTimerCallback elapsedCallback )
		{
			if( interval == null )
				throw( new ArgumentNullException( "interval" ) );

			if( elapsedCallback == null )
				throw( new ArgumentNullException( "elapsedCallback" ) );

			ResetStoppedFlag();
		
			_elapsedCallback = elapsedCallback;

			_timer = new System.Timers.Timer( interval.ToInt32Milliseconds() ) {AutoReset = autoReset};
		    _timer.Elapsed += new ElapsedEventHandler( ElapsedHandler );
		}

		/// <summary>
		/// Method resets 'stopped' timer state.
		/// </summary>
		private void ResetStoppedFlag()
		{
			lock( this )
				_stoppedFlag = false;
		}

		/// <summary>
		/// Stops the timer. Doesn't have any effect if timer is already stopped. 
		/// </summary>
		public void Stop()
		{
			lock( this )
			{
				_stoppedFlag = true;
				_timer.Stop();
			}
		}

		/// <summary>
		/// Stops and closes the timer.
		/// </summary>
		public void Close()
		{
			this.Stop();

			lock( this )				
				_timer.Close();
		}

		/// <summary>
		/// Starts the timer. Doesn't have any effect if timer is already started.
		/// </summary>
		public void Start()
		{
			lock( this )
			{
				if( _timer.Enabled ) 
					return;

				ResetStoppedFlag();
				_timer.Start();				
			}
		}

		/// <summary>
		/// Method handling timeout event. It stops/starts the timer if necessary 
		/// and calls callback method specified in constructor.
		/// </summary>
		/// <param name="sender">Object that invoked the method.</param>
		/// <param name="e">Event arguments.</param>
		private void ElapsedHandler( object sender, ElapsedEventArgs e )
		{
			lock( this )
			{
				if( _stoppedFlag )
					return;

				if( _timer.AutoReset ) 
					Stop();

				_elapsedCallback();

				if( _timer.AutoReset ) 
					Start();
			}
		}

		/// <summary>
		/// Flag indicating whether timer is enabled.
		/// </summary>
		public bool IsEnabled
		{
			get
			{
				lock( this )
					return _timer.Enabled;
			}
		}

		/// <summary>
		/// Timeout interval.
		/// </summary>
		public Timeout Interval
		{
			get
			{
				return new Timeout( Convert.ToInt32( _timer.Interval ) );
			}
			set
			{
				if( value == null )
					throw( new ArgumentNullException() );

				lock( this )
					_timer.Interval = value.ToInt32Milliseconds();
			}
		}
	}
}
